import fitz  # PyMuPDF
import pdfplumber
from PIL import Image
import io
import base64
from sentence_transformers import SentenceTransformer, util
import torch
import streamlit as st

def extract_text_from_pdf(pdf_path):
    with pdfplumber.open(pdf_path) as pdf:
        text_by_page = []
        for page in pdf.pages:
            text_by_page.append(page.extract_text())
    return text_by_page

def extract_images_from_pdf(pdf_path):
    doc = fitz.open(pdf_path)
    images = []
    for i in range(len(doc)):
        page = doc.load_page(i)
        for img in page.get_images(full=True):
            xref = img[0]
            base_image = doc.extract_image(xref)
            image_bytes = base_image["image"]
            image_ext = base_image["ext"]
            image = Image.open(io.BytesIO(image_bytes))
            images.append({
                "page": i,
                "xref": xref,
                "image": image,
                "bbox": img[3]  # bbox gives the bounding box of the image
            })
    return images

def display_image(image):
    try:
        buf = io.BytesIO()
        image.save(buf, format='PNG')
        byte_im = buf.getvalue()
        base64_img = base64.b64encode(byte_im).decode('utf-8')
        return base64_img
    except Exception as e:
        st.error(f"Error processing image: {e}")
        return None

def find_text_with_question(text_pages, question, model):
    question_embedding = model.encode(question, convert_to_tensor=True)
    highest_score = -1
    best_match = None
    best_page_number = -1

    for page_number, page_text in enumerate(text_pages):
        if page_text is None:
            continue
        page_sentences = page_text.split('\n')
        page_embeddings = model.encode(page_sentences, convert_to_tensor=True)
        scores = util.pytorch_cos_sim(question_embedding, page_embeddings)[0]
        max_score = torch.max(scores).item()

        if max_score > highest_score:
            highest_score = max_score
            best_match = page_sentences[scores.argmax().item()]
            best_page_number = page_number

    if best_match:
        paragraphs = text_pages[best_page_number].split('\n\n')  # Split text by paragraphs
        for paragraph in paragraphs:
            if best_match in paragraph:
                return paragraph, best_page_number

    return None, None

# Update the path to use raw string or double backslashes
pdf_path = r"pdf\Admin Panel Dashboard.pdf"  # Change this path as needed

text_by_page = extract_text_from_pdf(pdf_path)
images = extract_images_from_pdf(pdf_path)

model = SentenceTransformer('paraphrase-MiniLM-L6-v2')

st.title("NexsBot")

# Display the content of the first page
if 'first_page_displayed' not in st.session_state:
    st.session_state.first_page_displayed = False

if not st.session_state.first_page_displayed:
    if text_by_page and images:
        
        st.write(text_by_page[0])

        
        first_page_images = [img for img in images if img['page'] == 0]
        if first_page_images:
            image_base64 = display_image(first_page_images[0]['image'])
            if image_base64:
                st.image(f"data:image/png;base64,{image_base64}",  use_column_width=True)

    st.session_state.first_page_displayed = True
    

# Initialize chat state
if 'conversation' not in st.session_state:
    st.session_state.conversation = []
if 'question' not in st.session_state:
    st.session_state.question = ""
if 'show_input' not in st.session_state:
    st.session_state.show_input = True



def process_question():
    question = st.session_state.question
    if question.lower() == 'quit':
        st.session_state.show_input = False
        st.session_state.conversation.append({"text": "Thank you for using the chatbot!", "type": "bot"})
        st.session_state.question = ""
    else:
        if question:
            st.session_state.conversation.append({"text": question, "type": "user"})
            
            relevant_text, page_number = find_text_with_question(text_by_page, question, model)
            if relevant_text:
                combined_message = {"text": relevant_text, "type": "bot", "image": None}
                for image_info in images:
                    if image_info['page'] == page_number:
                        img = display_image(image_info['image'])
                        combined_message["image"] = img
                        break  # Only display the first image found on the page
                st.session_state.conversation.append(combined_message)
            else:
                st.session_state.conversation.append({"text": "No relevant text found for the question.", "type": "bot"})
            st.session_state.show_input = True
            st.session_state.question = ""
        else:
            st.session_state.conversation.append({"text": "Please enter a question.", "type": "bot"})
            st.session_state.show_input = True

st.markdown(
    """
    <style>
    .chat-box {
        border: 1px solid #ccc;
        padding: 10px;
        margin: 10px 0;
        border-radius: 10px;
        background: #f9f9f9;
    }
    .user-message {
        color: #333;
        font-weight: bold;
        margin-bottom: 5px;
        font-size: 1.7em
    }
    .bot-message {
        color: #007bff;
        margin-top: 5px;
    }
    .image-box {
        text-align: center;
        margin-top: 10px;
    }
    </style>
    """, unsafe_allow_html=True)

for i in range(0, len(st.session_state.conversation), 2):
    user_msg = st.session_state.conversation[i]
    bot_msg = st.session_state.conversation[i + 1] if i + 1 < len(st.session_state.conversation) else None
    
    st.markdown(
        f"""
        <div class="chat-box">
            <div class="user-message"><strong></strong> {user_msg['text']}</div>
            {"<div class='bot-message'><strong></strong> " + bot_msg['text'] + "</div>" if bot_msg and 'text' in bot_msg else ""}
            {"<div class='image-box'><img src='data:image/png;base64," + bot_msg['image'] + "' alt='Associated image'></div>" if bot_msg and 'image' in bot_msg else ""}
        </div>
        """, unsafe_allow_html=True)

if st.session_state.first_page_displayed and st.session_state.show_input:
    user_input = st.text_input("", key="question",placeholder="Type your message", on_change=process_question)
else:
    st.write("Thank you for using the chatbot!")



